const baudRates = [
  300, 600, 1200, 1800, 2400, 4800, 7200, 9600, 14400, 19200,
  28800, 31250, 38400, 56000, 57600, 74880, 115200, 128000,
  230400, 250000, 460800, 500000, 576000, 921600, 1000000,
  1152000, 1500000, 2000000, 2500000, 3000000
];

const gsmPresets = {
  simcom: [
    'AT', 'ATE0', 'ATI', 'AT+CPIN?', 'AT+CSQ', 'AT+CREG?', 'AT+CGATT?',
    'AT+COPS?', 'AT+CGDCONT?', 'AT+CMGF=1', 'AT+CMGL="ALL"', 'AT+CMGS="+1234567890"',
    'AT+CGSN', 'AT+CCID', 'AT+CBC'
  ],
  quectel: [
    'AT', 'ATE0', 'ATI', 'AT+CPIN?', 'AT+CSQ', 'AT+QCCID', 'AT+CGSN',
    'AT+CREG?', 'AT+QNWINFO', 'AT+QCSQ', 'AT+QCFG?', 'AT+CMGF=1',
    'AT+CMGL="ALL"', 'AT+QIACT?', 'AT+QENG="servingcell"'
  ]
};

const MAX_STORED_LOG_ROWS = 50000;
const MAX_STORED_PANEL_ROWS = 12000;
const MAX_RENDER_LOG_LINES = 1500;
const MAX_RENDER_PANEL_LINES = 1200;
const UI_RENDER_THROTTLE_MS = 80;

const state = {
  ports: [],
  panelCount: 1,
  serialPanels: [],
  logs: []
};

let logRenderTimer = null;
const panelRenderTimers = new Map();

const dom = {
  tabs: [...document.querySelectorAll('.tab')],
  pages: [...document.querySelectorAll('.page')],
  refreshPortsBtn: document.getElementById('refreshPortsBtn'),
  exportLogBtn: document.getElementById('exportLogBtn'),
  clearLogBtn: document.getElementById('clearLogBtn'),
  serialPanelCount: document.getElementById('serialPanelCount'),
  applyPanelCountBtn: document.getElementById('applyPanelCountBtn'),
  serialGrid: document.getElementById('serialGrid'),
  gsmPort: document.getElementById('gsmPort'),
  gsmConnection: document.getElementById('gsmConnection'),
  gsmVendor: document.getElementById('gsmVendor'),
  gsmCustomEnable: document.getElementById('gsmCustomEnable'),
  gsmCustomInput: document.getElementById('gsmCustomInput'),
  gsmRepeat: document.getElementById('gsmRepeat'),
  gsmDelay: document.getElementById('gsmDelay'),
  gsmVerifyEnable: document.getElementById('gsmVerifyEnable'),
  gsmExpected: document.getElementById('gsmExpected'),
  gsmSendCustom: document.getElementById('gsmSendCustom'),
  gsmCommands: document.getElementById('gsmCommands'),
  modbusPort: document.getElementById('modbusPort'),
  modbusBaud: document.getElementById('modbusBaud'),
  modbusSlave: document.getElementById('modbusSlave'),
  modbusRegister: document.getElementById('modbusRegister'),
  modbusLength: document.getElementById('modbusLength'),
  modbusType: document.getElementById('modbusType'),
  modbusReadBtn: document.getElementById('modbusReadBtn'),
  modbusWriteRegister: document.getElementById('modbusWriteRegister'),
  modbusWriteValue: document.getElementById('modbusWriteValue'),
  modbusWriteBtn: document.getElementById('modbusWriteBtn'),
  modbusRawHexInput: document.getElementById('modbusRawHexInput'),
  modbusRawHexSendBtn: document.getElementById('modbusRawHexSendBtn'),
  modbusOutput: document.getElementById('modbusOutput'),
  logConnectionFilter: document.getElementById('logConnectionFilter'),
  logDirectionFilter: document.getElementById('logDirectionFilter'),
  logModeFilter: document.getElementById('logModeFilter'),
  logOutput: document.getElementById('logOutput'),
  statusLine: document.getElementById('statusLine')
};

function setStatus(message) {
  dom.statusLine.textContent = `${new Date().toLocaleString()} - ${message}`;
}

function formatPort(port) {
  const details = [port.path];
  if (port.manufacturer) details.push(port.manufacturer);
  if (port.vendorId || port.productId) details.push(`VID:${port.vendorId || '-'} PID:${port.productId || '-'}`);
  return details.join(' | ');
}

function createPanelState(index) {
  return {
    slot: index,
    connectionId: `SERIAL-${index}`,
    open: false,
    path: '',
    baudRate: 9600,
    ioRows: [],
    optionsVisible: true
  };
}

function getPanelCard(panel) {
  return dom.serialGrid.querySelector(`[data-slot='${panel.slot}']`);
}

function setActivePage(pageKey) {
  for (const tab of dom.tabs) tab.classList.toggle('active', tab.dataset.tab === pageKey);
  for (const page of dom.pages) page.classList.toggle('active', page.id === `page-${pageKey}`);
}

function getOpenPanels() {
  return state.serialPanels.filter((p) => p.open);
}

function updateGsmConnectionOptions() {
  const open = getOpenPanels();
  const current = dom.gsmConnection.value;
  dom.gsmConnection.innerHTML = '';
  if (!open.length) {
    dom.gsmConnection.appendChild(new Option('No active serial', ''));
    return;
  }
  for (const panel of open) dom.gsmConnection.appendChild(new Option(`${panel.connectionId} (${panel.path})`, panel.connectionId));
  if ([...dom.gsmConnection.options].some((o) => o.value === current)) dom.gsmConnection.value = current;

  const selectedConn = open.find((p) => p.connectionId === dom.gsmConnection.value);
  if (selectedConn && dom.gsmPort) {
    dom.gsmPort.value = selectedConn.path;
  }
}

function updateGsmPortOptions() {
  const current = dom.gsmPort.value;
  dom.gsmPort.innerHTML = '';
  if (!state.ports.length) {
    dom.gsmPort.appendChild(new Option('No ports found', ''));
    return;
  }
  for (const port of state.ports) dom.gsmPort.appendChild(new Option(formatPort(port), port.path));
  if ([...dom.gsmPort.options].some((o) => o.value === current)) dom.gsmPort.value = current;
}

function updateModbusPortOptions() {
  const current = dom.modbusPort.value;
  dom.modbusPort.innerHTML = '';
  if (!state.ports.length) {
    dom.modbusPort.appendChild(new Option('No ports found', ''));
    return;
  }
  for (const port of state.ports) dom.modbusPort.appendChild(new Option(formatPort(port), port.path));
  if ([...dom.modbusPort.options].some((o) => o.value === current)) dom.modbusPort.value = current;
}

function updateLogConnectionFilterOptions() {
  const current = dom.logConnectionFilter.value;
  dom.logConnectionFilter.innerHTML = '';
  dom.logConnectionFilter.appendChild(new Option('All', 'ALL'));
  for (const panel of state.serialPanels) dom.logConnectionFilter.appendChild(new Option(panel.connectionId, panel.connectionId));
  if ([...dom.logConnectionFilter.options].some((o) => o.value === current)) dom.logConnectionFilter.value = current;
}

function appendLog(row) {
  state.logs.push(row);
  if (state.logs.length > MAX_STORED_LOG_ROWS) state.logs = state.logs.slice(-MAX_STORED_LOG_ROWS);
  scheduleLogRefresh();
}

function refreshLogOutput() {
  const c = dom.logConnectionFilter.value;
  const d = dom.logDirectionFilter.value;
  const m = dom.logModeFilter.value;
  const lines = state.logs
    .filter((r) => c === 'ALL' || r.connectionId === c)
    .filter((r) => d === 'ALL' || r.direction === d)
    .filter((r) => m === 'ALL' || r.mode === m)
    .slice(-MAX_RENDER_LOG_LINES)
    .map((r) => `[${r.timestamp}] [${r.connectionId}] [${r.direction}] [${r.mode}] ${r.payload}`);

  dom.logOutput.value = lines.join('\n');
  dom.logOutput.scrollTop = dom.logOutput.scrollHeight;
}

function scheduleLogRefresh(force = false) {
  const logsPageActive = document.getElementById('page-logs')?.classList.contains('active');
  if (!force && !logsPageActive) return;
  if (logRenderTimer) return;
  logRenderTimer = setTimeout(() => {
    logRenderTimer = null;
    refreshLogOutput();
  }, UI_RENDER_THROTTLE_MS);
}

function getPanelFilteredRows(panel) {
  const card = getPanelCard(panel);
  if (!card) return panel.ioRows;
  const dirFilter = card.querySelector('.io-dir').value;
  const modeFilter = card.querySelector('.io-mode').value;
  return panel.ioRows
    .filter((r) => dirFilter === 'ALL' || r.direction === dirFilter)
    .filter((r) => modeFilter === 'ALL' || r.mode === modeFilter);
}

function refreshPanelIo(panel) {
  const card = getPanelCard(panel);
  if (!card) return;
  const output = card.querySelector('.io-output');
  const rows = getPanelFilteredRows(panel);
  output.value = rows
    .slice(-MAX_RENDER_PANEL_LINES)
    .map((r) => `[${r.timestamp}] [${r.direction}] [${r.mode}] ${r.payload}`)
    .join('\n');
  if (card.querySelector('.io-autoscroll').checked) output.scrollTop = output.scrollHeight;
}

function schedulePanelRefresh(panel, force = false) {
  const key = panel.connectionId;
  if (!force && panelRenderTimers.has(key)) return;
  if (force) {
    const timer = panelRenderTimers.get(key);
    if (timer) {
      clearTimeout(timer);
      panelRenderTimers.delete(key);
    }
    refreshPanelIo(panel);
    return;
  }

  panelRenderTimers.set(
    key,
    setTimeout(() => {
      panelRenderTimers.delete(key);
      refreshPanelIo(panel);
    }, UI_RENDER_THROTTLE_MS)
  );
}

async function exportPanelTxt(panel) {
  const rows = state.logs.filter((r) => r.connectionId === panel.connectionId);
  if (!rows.length) return setStatus(`${panel.connectionId} has no rows to export.`);
  const result = await window.serialApi.exportLogTxt({ rows, filePrefix: panel.connectionId.toLowerCase() });
  if (result.canceled) return setStatus('TXT export canceled.');
  setStatus(`${panel.connectionId} exported: ${result.filePath}`);
}

function optionToggleLabel(visible) {
  return visible ? '[ - ] Hide Options' : '[ + ] Show Options';
}

function serialCardTemplate(panel) {
  const card = document.createElement('section');
  card.className = 'serial-card';
  card.dataset.slot = panel.slot;

  const baudOptions = baudRates.map((b) => `<option value='${b}' ${panel.baudRate === b ? 'selected' : ''}>${b}</option>`).join('');
  const portOptions = state.ports.length
    ? state.ports.map((p) => `<option value='${p.path}' ${panel.path === p.path ? 'selected' : ''}>${formatPort(p)}</option>`).join('')
    : `<option value=''>No ports found</option>`;

  card.innerHTML = `
    <div class="card-head">
      <h3>${panel.connectionId}</h3>
      <div class="row compact no-bottom">
        <span class="status-chip">${panel.open ? `Open @ ${panel.baudRate}` : 'Closed'}</span>
        <button class="btn-options-toggle quiet">${optionToggleLabel(panel.optionsVisible)}</button>
      </div>
    </div>

    <div class="options-body ${panel.optionsVisible ? '' : 'hidden'}">
      <div class="row compact">
        <label>COM Port
          <select class="cfg-port">${portOptions}</select>
        </label>
        <label>Baud
          <select class="cfg-baud">${baudOptions}</select>
        </label>
        <label>Custom Baud
          <input class="cfg-custom-baud" type="number" min="300" placeholder="Optional" />
        </label>
        <label>Data Bits
          <select class="cfg-data-bits">
            <option value="8">8</option>
            <option value="7">7</option>
            <option value="6">6</option>
            <option value="5">5</option>
          </select>
        </label>
        <label>Stop Bits
          <select class="cfg-stop-bits">
            <option value="1">1</option>
            <option value="2">2</option>
          </select>
        </label>
        <label>Parity
          <select class="cfg-parity">
            <option value="none">None</option>
            <option value="even">Even</option>
            <option value="odd">Odd</option>
            <option value="mark">Mark</option>
            <option value="space">Space</option>
          </select>
        </label>
        <button class="btn-open">Open</button>
        <button class="btn-close secondary">Close</button>
      </div>

      <div class="tx-toolbar compact">
        <strong>TX</strong>
        <label>Mode
          <select class="tx-mode">
            <option value="STRING">STRING</option>
            <option value="HEX">HEX</option>
          </select>
        </label>
        <label>Ending
          <select class="tx-ending">
            <option value="NONE">None</option>
            <option value="CR">CR</option>
            <option value="LF">LF</option>
            <option value="CRLF">CRLF</option>
          </select>
        </label>
        <label>Count <input type="number" class="tx-repeat" min="1" value="1" /></label>
        <label>Delay ms <input type="number" class="tx-delay" min="0" value="0" /></label>
        <label><input type="checkbox" class="tx-verify" /> Verify</label>
        <label>Expected <input type="text" class="tx-expected" placeholder="OK" /></label>
        <label>Timeout <input type="number" class="tx-timeout" min="100" value="1000" /></label>
        <label>Retries <input type="number" class="tx-retries" min="0" value="0" /></label>
        <input class="tx-input-line" type="text" placeholder="TX data" />
        <button class="btn-send">Send</button>
      </div>
    </div>

    <div class="io-toolbar">
      <strong>TX / RX Log</strong>
      <label>Direction
        <select class="io-dir">
          <option value="ALL">ALL</option>
          <option value="RX">RX</option>
          <option value="TX">TX</option>
        </select>
      </label>
      <label>View
        <select class="io-mode">
          <option value="ALL">ALL</option>
          <option value="TEXT">TEXT</option>
          <option value="HEX">HEX</option>
          <option value="RAW">RAW</option>
        </select>
      </label>
      <label><input type="checkbox" class="io-autoscroll" checked /> Auto scroll</label>
      <button class="btn-io-clear quiet">Clear</button>
      <button class="btn-export-txt quiet">Export TXT</button>
    </div>

    <textarea class="io-output" readonly></textarea>
  `;

  card.querySelector('.io-dir').addEventListener('change', () => refreshPanelIo(panel));
  card.querySelector('.io-mode').addEventListener('change', () => refreshPanelIo(panel));

  card.querySelector('.btn-options-toggle').addEventListener('click', () => {
    panel.optionsVisible = !panel.optionsVisible;
    card.querySelector('.options-body').classList.toggle('hidden', !panel.optionsVisible);
    card.querySelector('.btn-options-toggle').textContent = optionToggleLabel(panel.optionsVisible);
  });

  card.querySelector('.btn-io-clear').addEventListener('click', () => {
    panel.ioRows = [];
    schedulePanelRefresh(panel, true);
    setStatus(`${panel.connectionId} panel log cleared.`);
  });

  card.querySelector('.btn-export-txt').addEventListener('click', () => exportPanelTxt(panel));

  card.querySelector('.btn-open').addEventListener('click', async () => {
    try {
      const path = card.querySelector('.cfg-port').value;
      const customBaud = Number(card.querySelector('.cfg-custom-baud').value || 0);
      const baudRate = customBaud || Number(card.querySelector('.cfg-baud').value);
      const dataBits = Number(card.querySelector('.cfg-data-bits').value);
      const stopBits = Number(card.querySelector('.cfg-stop-bits').value);
      const parity = card.querySelector('.cfg-parity').value;
      if (!path) return setStatus('Select a COM port first.');
      const conflict = state.serialPanels.find((p) => p.connectionId !== panel.connectionId && p.open && p.path === path);
      if (conflict) return setStatus(`${path} already opened by ${conflict.connectionId}.`);

      const openResult = await window.serialApi.open({ connectionId: panel.connectionId, path, baudRate, dataBits, stopBits, parity });
      panel.open = true;
      panel.path = path;
      panel.baudRate = baudRate;
      card.querySelector('.status-chip').textContent = `Open @ ${baudRate}`;
      updateGsmConnectionOptions();
      if (openResult?.alreadyOpen) {
        setStatus(`${panel.connectionId} already connected to ${path}.`);
      } else {
        setStatus(`${panel.connectionId} connected to ${path}`);
      }
    } catch (err) {
      setStatus(`Open failed: ${err.message}`);
    }
  });

  card.querySelector('.btn-close').addEventListener('click', async () => {
    try {
      await window.serialApi.close({ connectionId: panel.connectionId });
      panel.open = false;
      card.querySelector('.status-chip').textContent = 'Closed';
      updateGsmConnectionOptions();
      setStatus(`${panel.connectionId} closed.`);
    } catch (err) {
      setStatus(`Close failed: ${err.message}`);
    }
  });

  card.querySelector('.btn-send').addEventListener('click', async () => {
    try {
      if (!panel.open) return setStatus(`Open ${panel.connectionId} before TX.`);
      const result = await window.serialApi.write({
        connectionId: panel.connectionId,
        mode: card.querySelector('.tx-mode').value,
        data: card.querySelector('.tx-input-line').value,
        lineEnding: card.querySelector('.tx-ending').value,
        repeat: Number(card.querySelector('.tx-repeat').value),
        delayMs: Number(card.querySelector('.tx-delay').value),
        verifyResponse: card.querySelector('.tx-verify').checked,
        expectedResponse: card.querySelector('.tx-expected').value,
        verifyTimeoutMs: Number(card.querySelector('.tx-timeout').value),
        maxRetries: Number(card.querySelector('.tx-retries').value)
      });
      setStatus(`${panel.connectionId} TX complete: ${result.sent} sent, ${result.retries} retries.`);
    } catch (err) {
      setStatus(`TX failed: ${err.message}`);
    }
  });

  return card;
}

function renderSerialPanels() {
  dom.serialGrid.style.setProperty('--serial-cols', String(Math.min(state.panelCount, 2)));
  dom.serialGrid.innerHTML = '';
  for (const panel of state.serialPanels) {
    dom.serialGrid.appendChild(serialCardTemplate(panel));
    schedulePanelRefresh(panel, true);
  }
  updateGsmConnectionOptions();
  updateLogConnectionFilterOptions();
}

async function applyPanelCount() {
  const target = Number(dom.serialPanelCount.value);
  if (target === state.panelCount) return;

  if (target < state.panelCount) {
    const closing = state.serialPanels.filter((p) => p.slot > target && p.open);
    for (const panel of closing) await window.serialApi.close({ connectionId: panel.connectionId });
  }

  state.panelCount = target;
  state.serialPanels = Array.from({ length: target }, (_, idx) => state.serialPanels[idx] || createPanelState(idx + 1));
  renderSerialPanels();
  setStatus(`Serial page configured for ${target} panel(s).`);
}

async function refreshPorts() {
  try {
    state.ports = await window.serialApi.listPorts();
    renderSerialPanels();
    updateGsmPortOptions();
    updateModbusPortOptions();
    setStatus(`${state.ports.length} serial port(s) detected.`);
  } catch (err) {
    setStatus(`Port scan failed: ${err.message}`);
  }
}

function renderGsmCommands() {
  const commands = gsmPresets[dom.gsmVendor.value] || [];
  dom.gsmCommands.innerHTML = '';
  for (const cmd of commands) {
    const btn = document.createElement('button');
    btn.textContent = cmd;
    btn.addEventListener('click', async () => {
      try {
        const connectionId = dom.gsmConnection.value;
        if (!connectionId) return setStatus('Select an active connection in GSM page.');
        await window.serialApi.write({ connectionId, mode: 'STRING', data: cmd, lineEnding: 'CRLF', repeat: 1, delayMs: 0, verifyResponse: false });
        setStatus(`GSM preset sent: ${cmd}`);
      } catch (err) {
        setStatus(`GSM preset failed: ${err.message}`);
      }
    });
    dom.gsmCommands.appendChild(btn);
  }
}

async function sendGsmCustom() {
  try {
    if (!dom.gsmCustomEnable.checked) return setStatus('Enable custom command first.');
    const connectionId = dom.gsmConnection.value;
    if (!connectionId) return setStatus('Select GSM connection first.');

    const result = await window.serialApi.write({
      connectionId,
      mode: 'STRING',
      data: dom.gsmCustomInput.value,
      lineEnding: 'CRLF',
      repeat: Number(dom.gsmRepeat.value),
      delayMs: Number(dom.gsmDelay.value),
      verifyResponse: dom.gsmVerifyEnable.checked,
      expectedResponse: dom.gsmExpected.value,
      verifyTimeoutMs: 1200,
      maxRetries: 2
    });
    setStatus(`GSM custom sent ${result.sent} time(s).`);
  } catch (err) {
    setStatus(`GSM custom failed: ${err.message}`);
  }
}

async function runModbusRead() {
  try {
    const data = await window.serialApi.modbusScan({
      path: dom.modbusPort.value,
      baudRate: Number(dom.modbusBaud.value),
      slaveId: Number(dom.modbusSlave.value),
      register: Number(dom.modbusRegister.value),
      length: Number(dom.modbusLength.value),
      functionCode: dom.modbusType.value
    });
    dom.modbusOutput.value = `Data: ${JSON.stringify(data.data)}\nRaw: ${data.buffer}`;
    setStatus('Modbus read completed.');
  } catch (err) {
    dom.modbusOutput.value = `Error: ${err.message}`;
    setStatus(`Modbus read failed: ${err.message}`);
  }
}

async function runModbusWrite() {
  try {
    const result = await window.serialApi.modbusWriteRegister({
      path: dom.modbusPort.value,
      baudRate: Number(dom.modbusBaud.value),
      slaveId: Number(dom.modbusSlave.value),
      register: Number(dom.modbusWriteRegister.value),
      value: Number(dom.modbusWriteValue.value)
    });
    dom.modbusOutput.value = `Write success: ${JSON.stringify(result.result)}`;
    setStatus('Modbus write completed.');
  } catch (err) {
    dom.modbusOutput.value = `Error: ${err.message}`;
    setStatus(`Modbus write failed: ${err.message}`);
  }
}

async function runModbusRawHex() {
  try {
    const result = await window.serialApi.modbusSendRawHex({
      path: dom.modbusPort.value,
      baudRate: Number(dom.modbusBaud.value),
      hex: dom.modbusRawHexInput.value
    });
    dom.modbusOutput.value = `Raw HEX sent: ${result.sentHex}`;
    setStatus('Modbus raw HEX sent.');
  } catch (err) {
    dom.modbusOutput.value = `Error: ${err.message}`;
    setStatus(`Modbus raw HEX failed: ${err.message}`);
  }
}

async function exportLogs() {
  if (!state.logs.length) return setStatus('No logs to export.');
  const result = await window.serialApi.exportLogTxt({ rows: state.logs, filePrefix: 'all-serial-logs' });
  if (result.canceled) return setStatus('TXT export canceled.');
  setStatus(`Logs exported to ${result.filePath}`);
}

function handleSerialEvent(evt) {
  const ts = evt.timestamp || new Date().toISOString();
  const panel = state.serialPanels.find((p) => p.connectionId === evt.connectionId);

  if ((evt.type === 'rx' || evt.type === 'rx-batch') && panel) {
    const rows = evt.type === 'rx-batch'
      ? (evt.rows || [])
      : [{ timestamp: ts, text: evt.text, rawHex: evt.rawHex }];

    for (const row of rows) {
      const rowTs = row.timestamp || ts;
      const raw = { timestamp: rowTs, connectionId: panel.connectionId, direction: 'RX', mode: 'RAW', payload: row.text || '' };
      const text = { timestamp: rowTs, connectionId: panel.connectionId, direction: 'RX', mode: 'TEXT', payload: row.text || '' };
      const hex = { timestamp: rowTs, connectionId: panel.connectionId, direction: 'RX', mode: 'HEX', payload: row.rawHex || '' };

      panel.ioRows.push(raw, text, hex);
      appendLog(raw);
      appendLog(text);
      appendLog(hex);
    }

    if (panel.ioRows.length > MAX_STORED_PANEL_ROWS) panel.ioRows = panel.ioRows.slice(-MAX_STORED_PANEL_ROWS);
    schedulePanelRefresh(panel);
  }

  if (evt.type === 'tx' && panel) {
    const txHex = { timestamp: ts, connectionId: panel.connectionId, direction: 'TX', mode: 'HEX', payload: evt.rawHex };
    const txText = { timestamp: ts, connectionId: panel.connectionId, direction: 'TX', mode: 'TEXT', payload: evt.text };

    panel.ioRows.push(txHex, txText);
    if (panel.ioRows.length > MAX_STORED_PANEL_ROWS) panel.ioRows = panel.ioRows.slice(-MAX_STORED_PANEL_ROWS);

    appendLog(txHex);
    appendLog(txText);
    schedulePanelRefresh(panel);
  }

  if (evt.type === 'closed' && panel) {
    panel.open = false;
    renderSerialPanels();
  }

  if (evt.type === 'status' || evt.type === 'error' || evt.type === 'closed' || evt.type === 'verify-miss') {
    setStatus(`[${evt.connectionId}] ${evt.message}`);
  }
}

function bindTabs() {
  for (const btn of dom.tabs) {
    btn.addEventListener('click', () => {
      setActivePage(btn.dataset.tab);
      if (btn.dataset.tab === 'logs') scheduleLogRefresh(true);
    });
  }
}

function bindEvents() {
  bindTabs();

  dom.refreshPortsBtn.addEventListener('click', refreshPorts);
  dom.exportLogBtn.addEventListener('click', exportLogs);
  dom.clearLogBtn.addEventListener('click', () => {
    state.logs = [];
    for (const panel of state.serialPanels) panel.ioRows = [];
    renderSerialPanels();
    scheduleLogRefresh(true);
    setStatus('All logs cleared.');
  });

  dom.applyPanelCountBtn.addEventListener('click', applyPanelCount);
  dom.logConnectionFilter.addEventListener('change', () => scheduleLogRefresh(true));
  dom.logDirectionFilter.addEventListener('change', () => scheduleLogRefresh(true));
  dom.logModeFilter.addEventListener('change', () => scheduleLogRefresh(true));

  dom.gsmVendor.addEventListener('change', renderGsmCommands);
  dom.gsmPort.addEventListener('change', () => {
    const selectedPath = dom.gsmPort.value;
    const panel = state.serialPanels.find((p) => p.open && p.path === selectedPath);
    if (panel) {
      dom.gsmConnection.value = panel.connectionId;
      setStatus(`GSM connection mapped to ${panel.connectionId} (${selectedPath}).`);
    } else if (selectedPath) {
      setStatus(`Selected GSM COM ${selectedPath} is not open. Open it in Raw Serial page first.`);
    }
  });
  dom.gsmConnection.addEventListener('change', () => {
    const panel = state.serialPanels.find((p) => p.connectionId === dom.gsmConnection.value);
    if (panel?.path) dom.gsmPort.value = panel.path;
  });
  dom.gsmCustomEnable.addEventListener('change', () => {
    dom.gsmCustomInput.disabled = !dom.gsmCustomEnable.checked;
  });
  dom.gsmSendCustom.addEventListener('click', sendGsmCustom);

  dom.modbusReadBtn.addEventListener('click', runModbusRead);
  dom.modbusWriteBtn.addEventListener('click', runModbusWrite);
  dom.modbusRawHexSendBtn.addEventListener('click', runModbusRawHex);

  window.serialApi.onEvent(handleSerialEvent);
}

async function init() {
  state.serialPanels = [createPanelState(1)];
  dom.serialPanelCount.value = String(state.panelCount);
  bindEvents();
  renderSerialPanels();
  renderGsmCommands();
  await refreshPorts();
  updateGsmPortOptions();
  scheduleLogRefresh(true);
  setActivePage('raw');
  setStatus('App initialized.');
}

init();
