const { app, BrowserWindow, ipcMain, dialog, Menu, shell } = require('electron');
const path = require('path');
const fs = require('fs');
const { SerialPort } = require('serialport');
const ModbusRTU = require('modbus-serial');

const connections = new Map();

function detachPortListeners(port) {
  try {
    port.removeAllListeners('data');
    port.removeAllListeners('error');
    port.removeAllListeners('close');
  } catch (_) {}
}

async function safeClosePort(port, timeoutMs = 1200) {
  await Promise.race([
    new Promise((resolve) => {
      if (!port?.isOpen) return resolve();
      port.close(() => resolve());
    }),
    new Promise((resolve) => setTimeout(resolve, timeoutMs))
  ]);

  if (port?.isOpen) {
    try {
      port.destroy();
    } catch (_) {}
  }
}

function findActiveByPath(portPath, excludeConnectionId = '') {
  for (const [id, state] of connections.entries()) {
    if (id === excludeConnectionId) continue;
    if (state.portPath === portPath && !state.stopped && (state.opening || state.port?.isOpen)) {
      return state;
    }
  }
  return null;
}

function stopConnectionTimers(state) {
  if (!state) return;
  try {
    if (state.rxFlushTimer) clearInterval(state.rxFlushTimer);
  } catch (_) {}
  state.rxFlushTimer = null;
}

function emitToRenderer(win, payload) {
  if (!win || win.isDestroyed()) return;
  win.webContents.send('serial:event', payload);
}

function safeBufferFromHex(hexText) {
  const clean = hexText.replace(/[^0-9a-fA-F]/g, '');
  if (!clean || clean.length % 2 !== 0) {
    throw new Error('Hex input must contain full byte pairs (e.g., AA55FF).');
  }
  return Buffer.from(clean, 'hex');
}

function lineEndingBytes(lineEnding) {
  switch (lineEnding) {
    case 'CR':
      return Buffer.from('\r');
    case 'LF':
      return Buffer.from('\n');
    case 'CRLF':
      return Buffer.from('\r\n');
    default:
      return Buffer.alloc(0);
  }
}

function createWindow() {
  const win = new BrowserWindow({
    width: 1440,
    height: 940,
    webPreferences: {
      preload: path.join(__dirname, 'preload.js'),
      contextIsolation: true,
      nodeIntegration: false
    }
  });

  win.loadFile(path.join(__dirname, 'index.html'));
}

function getHelpDataPath() {
  return path.join(__dirname, 'help-info.json');
}

function getDefaultHelpData() {
  return {
    sections: {
      learn_more: {
        title: 'Learn More',
        description: 'Add your product overview, getting started notes, and quick links.',
        items: [
          { title: 'Welcome', content: 'Edit help-info.json to customize this content.' }
        ]
      },
      documentation: {
        title: 'Documentation',
        description: 'Add user manuals, API notes, command references, and setup guides.',
        items: [
          { title: 'Serial Basics', content: 'Document connection setup, baud settings, and troubleshooting.' }
        ]
      },
      community_discussions: {
        title: 'Community Discussions',
        description: 'Add links to forums, groups, and internal discussion boards.',
        items: [
          { title: 'Community', content: 'Put your community link and discussion notes here.' }
        ]
      },
      search_issues: {
        title: 'Search Issues',
        description: 'List known issues, fixes, and escalation contacts.',
        items: [
          { title: 'Known Issues', content: 'Track known issues and workarounds in this section.' }
        ]
      }
    }
  };
}

function loadHelpData() {
  const filePath = getHelpDataPath();
  if (!fs.existsSync(filePath)) {
    fs.writeFileSync(filePath, JSON.stringify(getDefaultHelpData(), null, 2), 'utf8');
  }

  try {
    const raw = fs.readFileSync(filePath, 'utf8');
    return JSON.parse(raw);
  } catch (_) {
    return getDefaultHelpData();
  }
}

function sectionLabel(key, title) {
  if (title) return title;
  return String(key)
    .split('_')
    .map((s) => s.charAt(0).toUpperCase() + s.slice(1))
    .join(' ');
}

function buildHelpMenuSubmenu() {
  const helpData = loadHelpData();
  const sections = helpData?.sections || {};
  const order = ['learn_more', 'documentation', 'community_discussions', 'search_issues'];
  const keys = [...order.filter((k) => sections[k]), ...Object.keys(sections).filter((k) => !order.includes(k))];
  const submenu = [];

  for (const key of keys) {
    const section = sections[key] || {};
    const items = Array.isArray(section.items) ? section.items : [];

    if (!items.length) continue;

    submenu.push({
      label: sectionLabel(key, section.title),
      submenu: items.map((item) => ({
        label: item.title || 'Open',
        click: async () => {
          if (item.url) {
            await shell.openExternal(item.url);
            return;
          }
          await dialog.showMessageBox({
            type: 'info',
            title: item.title || sectionLabel(key, section.title),
            message: item.title || sectionLabel(key, section.title),
            detail: item.content || ''
          });
        }
      }))
    });
  }

  submenu.push({ type: 'separator' });
  submenu.push({
    label: 'Reload Help Menu',
    click: () => {
      Menu.setApplicationMenu(Menu.buildFromTemplate(buildAppMenuTemplate()));
    }
  });

  return submenu;
}

function buildAppMenuTemplate() {
  return [
    {
      label: 'File',
      submenu: [{ role: 'quit' }]
    },
    {
      label: 'Edit',
      submenu: [
        { role: 'undo' },
        { role: 'redo' },
        { type: 'separator' },
        { role: 'cut' },
        { role: 'copy' },
        { role: 'paste' },
        { role: 'delete' },
        { role: 'selectAll' }
      ]
    },
    {
      label: 'View',
      submenu: [{ role: 'reload' }, { role: 'forceReload' }, { type: 'separator' }, { role: 'toggleDevTools' }]
    },
    {
      label: 'Window',
      submenu: [{ role: 'minimize' }, { role: 'zoom' }, { type: 'separator' }, { role: 'close' }]
    },
    {
      label: 'Help',
      submenu: buildHelpMenuSubmenu()
    }
  ];
}

app.whenReady().then(() => {
  Menu.setApplicationMenu(Menu.buildFromTemplate(buildAppMenuTemplate()));
  createWindow();

  app.on('activate', () => {
    if (BrowserWindow.getAllWindows().length === 0) {
      createWindow();
    }
  });
});

app.on('window-all-closed', () => {
  for (const [id, state] of connections.entries()) {
    try {
      state.stopped = true;
      stopConnectionTimers(state);
      detachPortListeners(state.port);
      if (state.port?.isOpen) state.port.close();
    } catch (_) {}
    connections.delete(id);
  }

  if (process.platform !== 'darwin') app.quit();
});

ipcMain.handle('serial:listPorts', async () => {
  const ports = await SerialPort.list();
  return ports.map((p) => ({
    path: p.path,
    manufacturer: p.manufacturer || '',
    serialNumber: p.serialNumber || '',
    pnpId: p.pnpId || '',
    vendorId: p.vendorId || '',
    productId: p.productId || ''
  }));
});

ipcMain.handle('help:getContent', async () => {
  const filePath = getHelpDataPath();
  if (!fs.existsSync(filePath)) {
    fs.writeFileSync(filePath, JSON.stringify(getDefaultHelpData(), null, 2), 'utf8');
  }

  try {
    const raw = fs.readFileSync(filePath, 'utf8');
    const json = JSON.parse(raw);
    return { ok: true, data: json, filePath };
  } catch (error) {
    return { ok: false, error: `Invalid help-info.json: ${error.message}`, data: getDefaultHelpData(), filePath };
  }
});

ipcMain.handle('serial:open', async (event, cfg) => {
  const { connectionId, path: portPath, baudRate, dataBits, stopBits, parity } = cfg;

  if (!connectionId || !portPath) {
    throw new Error('Connection ID and COM port are required.');
  }

  const existing = connections.get(connectionId);
  if (existing) {
    if (!existing.stopped && (existing.opening || existing.port?.isOpen)) {
      if (existing.portPath === portPath) {
        return { ok: true, alreadyOpen: true };
      }
      throw new Error(`Connection ${connectionId} is already open. Close it before changing port.`);
    }
    existing.stopped = true;
    existing.closing = true;
    detachPortListeners(existing.port);
    await safeClosePort(existing.port);
    connections.delete(connectionId);
  }

  const inUseBy = findActiveByPath(portPath, connectionId);
  if (inUseBy) {
    throw new Error(`Port ${portPath} is already in use by ${inUseBy.connectionId}.`);
  }

  const win = BrowserWindow.fromWebContents(event.sender);

  const port = new SerialPort({
    path: portPath,
    baudRate: Number(baudRate),
    dataBits: Number(dataBits),
    stopBits: Number(stopBits),
    parity: String(parity),
    autoOpen: false
  });

  const state = {
    connectionId,
    portPath,
    rxBufferText: '',
    rxQueue: [],
    port,
    stopped: false,
    opening: true,
    closing: false,
    rxFlushTimer: null
  };

  connections.set(connectionId, state);

  port.on('data', (chunk) => {
    if (state.stopped || state.closing) return;
    const ts = new Date().toISOString();
    const text = chunk.toString('utf8');
    state.rxBufferText += text;
    if (state.rxBufferText.length > 500000) {
      state.rxBufferText = state.rxBufferText.slice(-500000);
    }

    state.rxQueue.push({
      timestamp: ts,
      rawHex: chunk.toString('hex').toUpperCase(),
      text
    });
    if (state.rxQueue.length > 5000) {
      state.rxQueue = state.rxQueue.slice(-5000);
    }
  });

  port.on('error', (err) => {
    if (state.stopped || state.closing) return;
    emitToRenderer(win, {
      type: 'error',
      connectionId,
      timestamp: new Date().toISOString(),
      message: err.message
    });
  });

  port.on('close', () => {
    state.stopped = true;
    state.opening = false;
    state.closing = false;
    stopConnectionTimers(state);
    state.rxQueue = [];
    connections.delete(connectionId);
    emitToRenderer(win, {
      type: 'closed',
      connectionId,
      timestamp: new Date().toISOString(),
      message: 'Port closed'
    });
  });

  try {
    await new Promise((resolve, reject) => {
      port.open((err) => (err ? reject(err) : resolve()));
    });
  } catch (err) {
    state.stopped = true;
    state.opening = false;
    state.closing = false;
    stopConnectionTimers(state);
    detachPortListeners(port);
    await safeClosePort(port);
    connections.delete(connectionId);
    throw err;
  }

  state.opening = false;
  state.rxFlushTimer = setInterval(() => {
    if (state.stopped || state.closing || !state.rxQueue.length) return;
    const rows = state.rxQueue.splice(0, state.rxQueue.length);
    emitToRenderer(win, {
      type: 'rx-batch',
      connectionId,
      rows
    });
  }, 40);

  emitToRenderer(win, {
    type: 'status',
    connectionId,
    timestamp: new Date().toISOString(),
    message: `Connected to ${portPath} @ ${baudRate}`
  });

  return { ok: true };
});

ipcMain.handle('serial:close', async (_event, { connectionId }) => {
  const state = connections.get(connectionId);
  if (!state) return { ok: true };

  state.stopped = true;
  state.opening = false;
  state.closing = true;
  stopConnectionTimers(state);
  state.rxQueue = [];
  detachPortListeners(state.port);
  await safeClosePort(state.port);

  connections.delete(connectionId);
  return { ok: true };
});

ipcMain.handle('serial:write', async (event, payload) => {
  const {
    connectionId,
    mode,
    data,
    lineEnding = 'NONE',
    repeat = 1,
    delayMs = 0,
    verifyResponse = false,
    expectedResponse = '',
    verifyTimeoutMs = 1000,
    maxRetries = 0
  } = payload;

  const state = connections.get(connectionId);
  if (!state || state.stopped || state.opening || state.closing || !state.port?.isOpen) {
    throw new Error(`Connection ${connectionId} is not open.`);
  }

  const win = BrowserWindow.fromWebContents(event.sender);

  const runWriteOnce = async () => {
    const content = mode === 'HEX' ? safeBufferFromHex(data) : Buffer.from(String(data), 'utf8');
    const bytes = Buffer.concat([content, lineEndingBytes(lineEnding)]);

    await new Promise((resolve, reject) => {
      state.port.write(bytes, (err) => {
        if (err) return reject(err);
        state.port.drain((drainErr) => (drainErr ? reject(drainErr) : resolve()));
      });
    });

    emitToRenderer(win, {
      type: 'tx',
      connectionId,
      timestamp: new Date().toISOString(),
      rawHex: bytes.toString('hex').toUpperCase(),
      text: bytes.toString('utf8')
    });
  };

  const waitForExpected = async () => {
    const target = expectedResponse || '';
    if (!target) return true;

    const startAt = Date.now();
    while (Date.now() - startAt < Number(verifyTimeoutMs)) {
      if ((state.rxBufferText || '').includes(target)) {
        return true;
      }
      await new Promise((r) => setTimeout(r, 50));
    }
    return false;
  };

  let sent = 0;
  let retries = 0;

  for (let i = 0; i < Number(repeat); i += 1) {
    let success = false;
    let attempts = 0;

    while (!success) {
      attempts += 1;
      await runWriteOnce();
      sent += 1;

      if (!verifyResponse) {
        success = true;
      } else {
        success = await waitForExpected();
        if (!success) {
          retries += 1;
          emitToRenderer(win, {
            type: 'verify-miss',
            connectionId,
            timestamp: new Date().toISOString(),
            message: `Expected response not matched for attempt ${attempts}`
          });
          if (attempts > Number(maxRetries) + 1) {
            throw new Error('Verification failed after max retries.');
          }
        }
      }

      if (!success && Number(delayMs) > 0) {
        await new Promise((r) => setTimeout(r, Number(delayMs)));
      }
    }

    if (Number(delayMs) > 0 && i < Number(repeat) - 1) {
      await new Promise((r) => setTimeout(r, Number(delayMs)));
    }
  }

  return { ok: true, sent, retries };
});

ipcMain.handle('log:export', async (_event, payload) => {
  const { rows } = payload;

  const { canceled, filePath } = await dialog.showSaveDialog({
    title: 'Export Serial Log',
    defaultPath: `serial-log-${Date.now()}.csv`,
    filters: [
      { name: 'CSV', extensions: ['csv'] },
      { name: 'Text', extensions: ['txt', 'log'] }
    ]
  });

  if (canceled || !filePath) return { ok: false, canceled: true };

  const ext = path.extname(filePath).toLowerCase();
  let content = '';

  if (ext === '.csv') {
    content = 'timestamp,connectionId,direction,mode,payload\n';
    content += rows
      .map((r) => [r.timestamp, r.connectionId, r.direction, r.mode, String(r.payload || '').replaceAll('"', '""')]
        .map((v) => `"${v}"`)
        .join(','))
      .join('\n');
  } else {
    content = rows
      .map((r) => `[${r.timestamp}] [${r.connectionId}] [${r.direction}] [${r.mode}] ${r.payload}`)
      .join('\n');
  }

  fs.writeFileSync(filePath, content, 'utf8');
  return { ok: true, filePath };
});

ipcMain.handle('log:exportTxt', async (_event, payload) => {
  const { rows, filePrefix = 'serial-log' } = payload;
  const { canceled, filePath } = await dialog.showSaveDialog({
    title: 'Export Serial Log (TXT)',
    defaultPath: `${filePrefix}-${Date.now()}.txt`,
    filters: [{ name: 'Text', extensions: ['txt'] }]
  });

  if (canceled || !filePath) return { ok: false, canceled: true };

  const content = rows
    .map((r) => `[${r.timestamp}] [${r.connectionId}] [${r.direction}] [${r.mode}] ${r.payload}`)
    .join('\n');

  fs.writeFileSync(filePath, content, 'utf8');
  return { ok: true, filePath };
});

ipcMain.handle('modbus:scan', async (_event, cfg) => {
  const {
    path: portPath,
    baudRate,
    parity = 'none',
    dataBits = 8,
    stopBits = 1,
    slaveId = 1,
    register = 0,
    length = 10,
    functionCode = 'holding'
  } = cfg;

  const client = new ModbusRTU();
  try {
    await client.connectRTUBuffered(portPath, {
      baudRate: Number(baudRate),
      parity,
      dataBits: Number(dataBits),
      stopBits: Number(stopBits)
    });
    client.setID(Number(slaveId));
    client.setTimeout(1500);

    let response;
    if (functionCode === 'holding') {
      response = await client.readHoldingRegisters(Number(register), Number(length));
    } else if (functionCode === 'input') {
      response = await client.readInputRegisters(Number(register), Number(length));
    } else if (functionCode === 'coils') {
      response = await client.readCoils(Number(register), Number(length));
    } else {
      response = await client.readDiscreteInputs(Number(register), Number(length));
    }

    return {
      ok: true,
      data: response?.data || [],
      buffer: response?.buffer?.toString('hex') || ''
    };
  } finally {
    try {
      client.close();
    } catch (_) {}
  }
});

ipcMain.handle('modbus:writeRegister', async (_event, cfg) => {
  const {
    path: portPath,
    baudRate,
    parity = 'none',
    dataBits = 8,
    stopBits = 1,
    slaveId = 1,
    register = 0,
    value = 0
  } = cfg;

  const client = new ModbusRTU();
  try {
    await client.connectRTUBuffered(portPath, {
      baudRate: Number(baudRate),
      parity,
      dataBits: Number(dataBits),
      stopBits: Number(stopBits)
    });
    client.setID(Number(slaveId));
    client.setTimeout(1500);
    const result = await client.writeRegister(Number(register), Number(value));
    return { ok: true, result };
  } finally {
    try {
      client.close();
    } catch (_) {}
  }
});

ipcMain.handle('modbus:sendRawHex', async (_event, cfg) => {
  const {
    path: portPath,
    baudRate,
    parity = 'none',
    dataBits = 8,
    stopBits = 1,
    hex = ''
  } = cfg;

  if (!portPath) throw new Error('COM port is required.');
  const bytes = safeBufferFromHex(String(hex));

  const port = new SerialPort({
    path: portPath,
    baudRate: Number(baudRate),
    parity: String(parity),
    dataBits: Number(dataBits),
    stopBits: Number(stopBits),
    autoOpen: false
  });

  await new Promise((resolve, reject) => {
    port.open((err) => (err ? reject(err) : resolve()));
  });

  try {
    await new Promise((resolve, reject) => {
      port.write(bytes, (err) => {
        if (err) return reject(err);
        port.drain((drainErr) => (drainErr ? reject(drainErr) : resolve()));
      });
    });
  } finally {
    await new Promise((resolve) => {
      if (!port.isOpen) return resolve();
      port.close(() => resolve());
    });
  }

  return { ok: true, sentHex: bytes.toString('hex').toUpperCase() };
});
