import threading
import time
import tkinter as tk
from tkinter import messagebox, ttk

import cv2
import numpy as np

try:
    import winsound
except ImportError:
    winsound = None

MODEL_OPTIONS = {
    "yolov5n": {"backend": "v5", "weights": "yolov5n"},
    "yolov5s": {"backend": "v5", "weights": "yolov5s"},
    "yolov5m": {"backend": "v5", "weights": "yolov5m"},
    "yolov5l": {"backend": "v5", "weights": "yolov5l"},
    "yolov5x": {"backend": "v5", "weights": "yolov5x"},
    "yolov8n": {"backend": "ultralytics", "weights": "yolov8n.pt"},
    "yolov8s": {"backend": "ultralytics", "weights": "yolov8s.pt"},
    "yolov8m": {"backend": "ultralytics", "weights": "yolov8m.pt"},
    "yolov8l": {"backend": "ultralytics", "weights": "yolov8l.pt"},
    "yolov8x": {"backend": "ultralytics", "weights": "yolov8x.pt"},
    "yolo11n": {"backend": "ultralytics", "weights": "yolo11n.pt"},
    "yolo11s": {"backend": "ultralytics", "weights": "yolo11s.pt"},
    "yolo11m": {"backend": "ultralytics", "weights": "yolo11m.pt"},
    "yolo11l": {"backend": "ultralytics", "weights": "yolo11l.pt"},
    "yolo11x": {"backend": "ultralytics", "weights": "yolo11x.pt"},
}


class DetectorApp:
    def __init__(self, root: tk.Tk) -> None:
        self.root = root
        self.root.title("YOLO Object and Person Detection")
        self.root.geometry("420x220")

        self.model_var = tk.StringVar(value="yolov8n")
        self.status_var = tk.StringVar(value="Ready")

        self.running = False
        self.detect_thread = None
        self.last_beep_time = 0.0
        self.beep_cooldown_seconds = 1.0

        self.model = None
        self.model_type = None
        self.model_backend = None

        self._build_ui()

    def _build_ui(self) -> None:
        frame = ttk.Frame(self.root, padding=16)
        frame.pack(fill=tk.BOTH, expand=True)

        title = ttk.Label(frame, text="Realtime Webcam Detection", font=("Segoe UI", 14, "bold"))
        title.pack(anchor=tk.W, pady=(0, 12))

        model_row = ttk.Frame(frame)
        model_row.pack(fill=tk.X, pady=(0, 12))

        ttk.Label(model_row, text="Select YOLO model:", width=20).pack(side=tk.LEFT)
        version_box = ttk.Combobox(
            model_row,
            textvariable=self.model_var,
            values=list(MODEL_OPTIONS.keys()),
            state="readonly",
            width=12,
        )
        version_box.pack(side=tk.LEFT)

        btn_row = ttk.Frame(frame)
        btn_row.pack(fill=tk.X, pady=(0, 12))

        self.start_btn = ttk.Button(btn_row, text="Start Detection", command=self.start_detection)
        self.start_btn.pack(side=tk.LEFT, padx=(0, 8))

        self.stop_btn = ttk.Button(btn_row, text="Stop Detection", command=self.stop_detection, state=tk.DISABLED)
        self.stop_btn.pack(side=tk.LEFT)

        ttk.Label(frame, textvariable=self.status_var, foreground="blue").pack(anchor=tk.W, pady=(10, 0))
        ttk.Label(
            frame,
            text="When a person is detected, a beep plays.\nPress Q in the camera window to stop.",
            foreground="#444",
        ).pack(anchor=tk.W, pady=(8, 0))

    def set_status(self, text: str) -> None:
        self.root.after(0, lambda: self.status_var.set(text))

    def _play_beep(self) -> None:
        now = time.time()
        if now - self.last_beep_time < self.beep_cooldown_seconds:
            return

        self.last_beep_time = now
        if winsound is not None:
            winsound.Beep(1200, 250)
        else:
            print("Person detected! (beep not supported on this OS)")

    def _load_model(self, model_name: str) -> None:
        if self.model is not None and self.model_type == model_name:
            return

        config = MODEL_OPTIONS.get(model_name)
        if config is None:
            raise ValueError(f"Unsupported model: {model_name}")

        self.set_status(f"Loading {model_name} model...")
        backend = config["backend"]
        weights = config["weights"]

        if backend == "v5":
            import torch

            # Downloads model weights on first run.
            self.model = torch.hub.load("ultralytics/yolov5", weights, pretrained=True)
            self.model_type = model_name
            self.model_backend = "v5"
        elif backend == "ultralytics":
            from ultralytics import YOLO

            # Downloads model weights on first run.
            self.model = YOLO(weights)
            self.model_type = model_name
            self.model_backend = "ultralytics"
        else:
            raise ValueError(f"Unsupported backend: {backend}")

    def _contains_person(self, frame) -> bool:
        if self.model_backend == "v5":
            results = self.model(frame)
            det = results.xyxy[0]
            if det is None or len(det) == 0:
                return False

            # Class ID 0 in COCO is person.
            class_ids = det[:, 5].tolist()
            return any(int(cls_id) == 0 for cls_id in class_ids)

        if self.model_backend == "ultralytics":
            results = self.model.predict(source=frame, conf=0.35, verbose=False)
            if not results:
                return False

            boxes = results[0].boxes
            if boxes is None or boxes.cls is None:
                return False

            class_ids = boxes.cls.tolist()
            return any(int(cls_id) == 0 for cls_id in class_ids)

        return False

    def _draw_predictions(self, frame):
        if self.model_backend == "v5":
            results = self.model(frame)
            rendered = np.ascontiguousarray(results.render()[0]).copy()
            return rendered, results

        if self.model_backend == "ultralytics":
            results = self.model.predict(source=frame, conf=0.35, verbose=False)
            plotted = np.ascontiguousarray(results[0].plot()).copy()
            return plotted, results

        return np.ascontiguousarray(frame).copy(), None

    def _detection_loop(self, model_name: str) -> None:
        cap = cv2.VideoCapture(0)
        if not cap.isOpened():
            self.set_status("Failed to open webcam")
            self.root.after(0, self._reset_buttons)
            return

        try:
            self._load_model(model_name)
            self.set_status(f"Running {model_name} detection...")

            while self.running:
                ok, frame = cap.read()
                if not ok:
                    self.set_status("Failed to read frame")
                    break

                output_frame, results = self._draw_predictions(frame)

                person_found = False
                if self.model_backend == "v5" and results is not None:
                    det = results.xyxy[0]
                    if det is not None and len(det) > 0:
                        class_ids = det[:, 5].tolist()
                        person_found = any(int(c) == 0 for c in class_ids)
                elif self.model_backend == "ultralytics" and results:
                    boxes = results[0].boxes
                    if boxes is not None and boxes.cls is not None:
                        person_found = any(int(c) == 0 for c in boxes.cls.tolist())

                if person_found:
                    self._play_beep()
                    cv2.putText(
                        output_frame,
                        "PERSON DETECTED",
                        (20, 40),
                        cv2.FONT_HERSHEY_SIMPLEX,
                        1.0,
                        (0, 0, 255),
                        2,
                    )

                cv2.imshow("YOLO Live Detection (press Q to quit)", output_frame)

                if cv2.waitKey(1) & 0xFF == ord("q"):
                    break
        except Exception as exc:
            self.set_status(f"Error: {exc}")
            messagebox.showerror("Detection Error", str(exc))
        finally:
            self.running = False
            cap.release()
            cv2.destroyAllWindows()
            self.root.after(0, self._reset_buttons)
            self.set_status("Stopped")

    def _reset_buttons(self) -> None:
        self.start_btn.config(state=tk.NORMAL)
        self.stop_btn.config(state=tk.DISABLED)

    def start_detection(self) -> None:
        if self.running:
            return

        model_name = self.model_var.get().strip()
        if model_name not in MODEL_OPTIONS:
            messagebox.showwarning("Invalid Selection", "Choose a valid model from dropdown.")
            return

        self.running = True
        self.start_btn.config(state=tk.DISABLED)
        self.stop_btn.config(state=tk.NORMAL)
        self.detect_thread = threading.Thread(target=self._detection_loop, args=(model_name,), daemon=True)
        self.detect_thread.start()

    def stop_detection(self) -> None:
        self.running = False
        self.set_status("Stopping...")


def main() -> None:
    root = tk.Tk()
    app = DetectorApp(root)
    root.protocol("WM_DELETE_WINDOW", lambda: (app.stop_detection(), root.destroy()))
    root.mainloop()


if __name__ == "__main__":
    main()
