/*
 * Copyright (c) 2020, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 *  ======== uart2echo.c ========
 */
#include <stdint.h>
#include <stddef.h>

/* Driver Header files */
#include <ti/drivers/GPIO.h>
#include <ti/drivers/UART2.h>

/* Driver configuration */
#include "ti_drivers_config.h"
#include <stdint.h>
#include <stdbool.h>
#include <ti/drivers/SPI.h>
#include <ti/drivers/GPIO.h>
#include <ti/drivers/dpl/ClockP.h>
#include "ti_drivers_config.h"
/*
 *  ======== mainThread ========
 */
static SPI_Handle dacSpi;
void DAC8551_init(void);
bool DAC8551_writeRegister(uint16_t value, uint8_t pd);
void *mainThread(void *arg0)
{
    SPI_init();
        GPIO_init();
        DAC8551_init();

        const int percents[] = {10, 40, 60, 100};
        const int nSteps = sizeof(percents) / sizeof(percents[0]);
        const int delay_ms = 20000; // 20 seconds
        const uint32_t FULL = 65535U;

        while (1) {
            for (int i = 0; i < nSteps; ++i) {
                uint32_t p = (uint32_t)percents[i];

                uint32_t code = (p * FULL + 50U) / 100U;
                if (code > FULL) code = FULL;

                DAC8551_writeRegister((uint16_t)code, 0); // pd = 0 (normal)
                ClockP_sleep(delay_ms / 1000);
                ClockP_usleep((delay_ms % 1000) * 1000);
            }
        }
}

void DAC8551_init(void)
{
    SPI_Params spiParams;
    SPI_Params_init(&spiParams);

    spiParams.frameFormat = SPI_POL0_PHA1;  // SPI Mode 1
    spiParams.dataSize    = 8;              // 8-bit data frames
    spiParams.bitRate     = 1000000;        // 1 MHz

    dacSpi = SPI_open(CONFIG_SPI_0, &spiParams);
    if (dacSpi == NULL) {
        while (1) { /* SPI init failed */ }
    }
}
bool DAC8551_writeRegister(uint16_t value, uint8_t pd)
{
    SPI_Transaction trans;
    uint8_t txBuf[3];

    uint32_t frame = ((uint32_t)(pd & 0x03) << 16) | (uint32_t)value;

    txBuf[0] = (uint8_t)((frame >> 16) & 0xFF);
    txBuf[1] = (uint8_t)((frame >> 8) & 0xFF);
    txBuf[2] = (uint8_t)(frame & 0xFF);

    trans.count = 3;
    trans.txBuf = txBuf;
    trans.rxBuf = NULL;

    return SPI_transfer(dacSpi, &trans);
}
