/* USER CODE BEGIN Header */
/**
  ******************************************************************************
  * @file           : main.c
  * @brief          : Main program body - GSM SMS GPIO Controller
  ******************************************************************************
  * @attention
  *
  * This project allows STM32 to control GPIO peripherals (like motors, lights,
  * PCs, etc.) via SMS commands such as "ON" or "OFF" through a Quectel GSM module.
  *
  * - STM32 Bluepill + Quectel EC200U (or similar)
  * - UART1: GSM Communication
  * - UART2: Debug Output
  * - GPIOB Pin 3: Example relay output (controlled by SMS)
  *
  * Copyright (c) 2025
  * All rights reserved.
  *
  ******************************************************************************
  */
/* USER CODE END Header */

/* Includes ------------------------------------------------------------------*/
#include "main.h"
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdbool.h>
#include <ctype.h>

/* Private variables ---------------------------------------------------------*/
ADC_HandleTypeDef hadc1;
TIM_HandleTypeDef htim2;
UART_HandleTypeDef huart1;   // GSM UART
UART_HandleTypeDef huart2;   // Debug UART

/* Private function prototypes -----------------------------------------------*/
void SystemClock_Config(void);
static void MX_GPIO_Init(void);
static void MX_ADC1_Init(void);
static void MX_USART1_UART_Init(void);
static void MX_USART2_UART_Init(void);
static void MX_TIM2_Init(void);
void extract_and_print_sms(const char *buffer);

/* USER CODE BEGIN 0 */

// UART1 (GSM) receive buffer variables
char uart1_rx_byte;
char uart1_buffer[512];
uint16_t uart1_index = 0;

// AT command list for GSM initialization
const char *at_commands[] = {
    "AT",               // Basic check
    "ATE0",             // Echo off
    "AT+CLIP=1",        // Caller ID enable
    "AT+CVHU=0",        // Voice call hangup disable
    "AT+CTZU=1",        // Time update enable
    "AT+CMGF=1",        // SMS text mode
    "AT+CSQ"            // Check signal quality
};
const int total_cmds = sizeof(at_commands) / sizeof(at_commands[0]);

/* -------------------------------------------------------------------------- */
/* UART TRANSMIT FUNCTIONS */
/* -------------------------------------------------------------------------- */

/**
 * @brief Send AT command to GSM module and print it on debug UART
 */
void sendAT(const char *cmd) {
    char debug_buf[128];
    snprintf(debug_buf, sizeof(debug_buf), "Sending: %s\r\n", cmd);
    HAL_UART_Transmit(&huart2, (uint8_t *)debug_buf, strlen(debug_buf), HAL_MAX_DELAY);

    char at_cmd[64];
    snprintf(at_cmd, sizeof(at_cmd), "%s\r\n", cmd);
    HAL_UART_Transmit(&huart1, (uint8_t *)at_cmd, strlen(at_cmd), HAL_MAX_DELAY);
}

/**
 * @brief Check signal quality from "+CSQ" response
 */
bool is_signal_ok(const char *csq) {
    int rssi = 0;
    if (sscanf(csq, "+CSQ: %d", &rssi) == 1) {
        return (rssi > 0 && rssi < 99);
    }
    return false;
}

/**
 * @brief Wait until GSM module reports acceptable signal strength
 */
void wait_for_signal(void) {
    char response[128];
    while (1) {
        sendAT("AT+CSQ");
        HAL_Delay(500);

        strncpy(response, uart1_buffer, sizeof(response));
        if (is_signal_ok(response)) {
            HAL_UART_Transmit(&huart2, (uint8_t *)"Signal OK\r\n", 11, HAL_MAX_DELAY);
            break;
        } else {
            HAL_UART_Transmit(&huart2, (uint8_t *)"Waiting for signal...\r\n", 24, HAL_MAX_DELAY);
            HAL_Delay(1000);
        }
    }
}

/* -------------------------------------------------------------------------- */
/* UART RECEIVE AND PROCESSING */
/* -------------------------------------------------------------------------- */

/**
 * @brief Handle received UART1 (GSM) data
 */
void process_uart1_data(char byte) {
    static bool last_byte_was_cr = false;

    // Echo incoming GSM data to debug UART
    HAL_UART_Transmit(&huart2, (uint8_t *)&byte, 1, HAL_MAX_DELAY);

    if (uart1_index < sizeof(uart1_buffer) - 1) {
        uart1_buffer[uart1_index++] = byte;
        uart1_buffer[uart1_index] = '\0';

        // End of line detected
        if (byte == '\n' && last_byte_was_cr) {
            // Check if an SMS arrived
            if (strstr(uart1_buffer, "+CMT:")) {
                extract_and_print_sms(uart1_buffer);
            }

            // Check for "OK" or "ERROR" completion
            if (strstr(uart1_buffer, "\r\nOK\r\n") || strstr(uart1_buffer, "\r\nERROR\r\n")) {
                char debug_msg[512];
                snprintf(debug_msg, sizeof(debug_msg), "\r\n[GSM]: %.500s\r\n", uart1_buffer);
                HAL_UART_Transmit(&huart2, (uint8_t *)debug_msg, strlen(debug_msg), HAL_MAX_DELAY);

                // Signal quality parsing
                if (strstr(uart1_buffer, "+CSQ:")) {
                    if (is_signal_ok(uart1_buffer))
                        HAL_UART_Transmit(&huart2, (uint8_t *)"Signal OK\r\n", 11, HAL_MAX_DELAY);
                    else
                        HAL_UART_Transmit(&huart2, (uint8_t *)"Weak Signal\r\n", 13, HAL_MAX_DELAY);
                }

                uart1_index = 0;
                memset(uart1_buffer, 0, sizeof(uart1_buffer));
            }
        }

        last_byte_was_cr = (byte == '\r');
    } else {
        uart1_index = 0;
        memset(uart1_buffer, 0, sizeof(uart1_buffer));
    }
}

/**
 * @brief UART RX Complete interrupt callback
 */
void HAL_UART_RxCpltCallback(UART_HandleTypeDef *huart) {
    if (huart->Instance == USART1) {
        process_uart1_data(uart1_rx_byte);
        HAL_UART_Receive_IT(&huart1, (uint8_t *)&uart1_rx_byte, 1);
    }
}

/* -------------------------------------------------------------------------- */
/* SMS EXTRACTION AND GPIO CONTROL */
/* -------------------------------------------------------------------------- */

/**
 * @brief Extract SMS message content and act on commands
 */
void extract_and_print_sms(const char *buffer) {
    const char *cmt_ptr = strstr(buffer, "+CMT:");
    if (!cmt_ptr) return;

    const char *line_end = strstr(cmt_ptr, "\r\n");
    if (!line_end) return;

    const char *msg_start = line_end + 2;
    const char *msg_end = strstr(msg_start, "\r\n");
    if (!msg_end) return;

    int msg_len = msg_end - msg_start;
    if (msg_len <= 0 || msg_len >= 256) return;

    char message[256];
    strncpy(message, msg_start, msg_len);
    message[msg_len] = '\0';

    // Trim spaces/newlines
    while (msg_len > 0 && (message[msg_len - 1] == '\r' || message[msg_len - 1] == '\n' || message[msg_len - 1] == ' '))
        message[--msg_len] = '\0';

    char debug_output[300];
    snprintf(debug_output, sizeof(debug_output), "\r\n[Received SMS]: %s\r\n", message);
    HAL_UART_Transmit(&huart2, (uint8_t *)debug_output, strlen(debug_output), HAL_MAX_DELAY);

    // Reset buffer
    uart1_index = 0;
    memset(uart1_buffer, 0, sizeof(uart1_buffer));

    // ACT ON SMS COMMANDS
    if (strcmp(message, "ON") == 0) {
        HAL_GPIO_WritePin(GPIOB, GPIO_PIN_3, GPIO_PIN_SET);    // Turn ON relay
        HAL_UART_Transmit(&huart2, (uint8_t *)"Relay ON\r\n", 10, HAL_MAX_DELAY);
    } 
    else if (strcmp(message, "OFF") == 0) {
        HAL_GPIO_WritePin(GPIOB, GPIO_PIN_3, GPIO_PIN_RESET);  // Turn OFF relay
        HAL_UART_Transmit(&huart2, (uint8_t *)"Relay OFF\r\n", 11, HAL_MAX_DELAY);
    }
}

/* USER CODE END 0 */

/* -------------------------------------------------------------------------- */
/* MAIN FUNCTION */
/* -------------------------------------------------------------------------- */
int main(void) {
    HAL_Init();
    SystemClock_Config();

    MX_GPIO_Init();
    MX_ADC1_Init();
    MX_USART1_UART_Init();
    MX_USART2_UART_Init();
    MX_TIM2_Init();

    // Start PWM timers (if needed)
    HAL_TIM_PWM_Start(&htim2, TIM_CHANNEL_1);
    HAL_TIM_PWM_Start(&htim2, TIM_CHANNEL_2);
    HAL_TIM_PWM_Start(&htim2, TIM_CHANNEL_3);
    HAL_TIM_PWM_Start(&htim2, TIM_CHANNEL_4);

    // Start GSM UART interrupt reception
    HAL_UART_Receive_IT(&huart1, (uint8_t *)&uart1_rx_byte, 1);

    HAL_UART_Transmit(&huart2, (uint8_t *)"\r\nSystem Started\r\n", 17, HAL_MAX_DELAY);

    // Send basic AT initialization commands
    for (int i = 0; i < total_cmds; i++) {
        sendAT(at_commands[i]);
        HAL_Delay(500);
    }

    // Optional: Wait for signal before continuing
    // wait_for_signal();

    // GSM SMS Configuration
    sendAT("AT+CREG?");                        HAL_Delay(500);
    sendAT("AT+CGREG?");                       HAL_Delay(500);
    sendAT("AT+CMGD=1,4");                     HAL_Delay(500); // Delete all SMS
    sendAT("AT+QURCCFG=\"urcport\",\"uart1\"");HAL_Delay(500);
    sendAT("AT+CNMI=2,2,0,0,0");               HAL_Delay(500); // Auto SMS receive to UART

    sendAT("System Ready...");

    // Main loop
    while (1) {
        HAL_Delay(500);
    }
}

/* -------------------------------------------------------------------------- */
/* Peripheral Initialization */
/* -------------------------------------------------------------------------- */

void SystemClock_Config(void) {
    RCC_OscInitTypeDef RCC_OscInitStruct = {0};
    RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};
    RCC_PeriphCLKInitTypeDef PeriphClkInit = {0};

    RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSE;
    RCC_OscInitStruct.HSEState = RCC_HSE_ON;
    RCC_OscInitStruct.HSEPredivValue = RCC_HSE_PREDIV_DIV1;
    RCC_OscInitStruct.HSIState = RCC_HSI_ON;
    RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
    RCC_OscInitStruct.PLL.PLLSource = RCC_PLLSOURCE_HSE;
    RCC_OscInitStruct.PLL.PLLMUL = RCC_PLL_MUL9;
    HAL_RCC_OscConfig(&RCC_OscInitStruct);

    RCC_ClkInitStruct.ClockType = RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_SYSCLK |
                                  RCC_CLOCKTYPE_PCLK1 | RCC_CLOCKTYPE_PCLK2;
    RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_PLLCLK;
    RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;
    RCC_ClkInitStruct.APB1CLKDivider = RCC_HCLK_DIV2;
    RCC_ClkInitStruct.APB2CLKDivider = RCC_HCLK_DIV1;
    HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_2);

    PeriphClkInit.PeriphClockSelection = RCC_PERIPHCLK_ADC;
    PeriphClkInit.AdcClockSelection = RCC_ADCPCLK2_DIV6;
    HAL_RCCEx_PeriphCLKConfig(&PeriphClkInit);

    HAL_RCC_MCOConfig(RCC_MCO, RCC_MCO1SOURCE_PLLCLK, RCC_MCODIV_1);
}

/* ADC Initialization */
static void MX_ADC1_Init(void) {
    ADC_ChannelConfTypeDef sConfig = {0};
    hadc1.Instance = ADC1;
    hadc1.Init.ScanConvMode = ADC_SCAN_DISABLE;
    hadc1.Init.ContinuousConvMode = DISABLE;
    hadc1.Init.ExternalTrigConv = ADC_SOFTWARE_START;
    hadc1.Init.DataAlign = ADC_DATAALIGN_RIGHT;
    hadc1.Init.NbrOfConversion = 1;
    HAL_ADC_Init(&hadc1);

    sConfig.Channel = ADC_CHANNEL_6;
    sConfig.Rank = ADC_REGULAR_RANK_1;
    sConfig.SamplingTime = ADC_SAMPLETIME_1CYCLE_5;
    HAL_ADC_ConfigChannel(&hadc1, &sConfig);
}

/* TIM2 (PWM) Initialization */
static void MX_TIM2_Init(void) {
    TIM_MasterConfigTypeDef sMasterConfig = {0};
    TIM_OC_InitTypeDef sConfigOC = {0};

    htim2.Instance = TIM2;
    htim2.Init.Prescaler = 0;
    htim2.Init.CounterMode = TIM_COUNTERMODE_UP;
    htim2.Init.Period = 65535;
    HAL_TIM_PWM_Init(&htim2);

    sMasterConfig.MasterOutputTrigger = TIM_TRGO_RESET;
    sMasterConfig.MasterSlaveMode = TIM_MASTERSLAVEMODE_DISABLE;
    HAL_TIMEx_MasterConfigSynchronization(&htim2, &sMasterConfig);

    sConfigOC.OCMode = TIM_OCMODE_PWM1;
    sConfigOC.Pulse = 0;
    sConfigOC.OCPolarity = TIM_OCPOLARITY_HIGH;
    sConfigOC.OCFastMode = TIM_OCFAST_DISABLE;

    HAL_TIM_PWM_ConfigChannel(&htim2, &sConfigOC, TIM_CHANNEL_1);
    HAL_TIM_PWM_ConfigChannel(&htim2, &sConfigOC, TIM_CHANNEL_2);
    HAL_TIM_PWM_ConfigChannel(&htim2, &sConfigOC, TIM_CHANNEL_3);
    HAL_TIM_PWM_ConfigChannel(&htim2, &sConfigOC, TIM_CHANNEL_4);

    HAL_TIM_MspPostInit(&htim2);
}

/* USART1 Initialization (GSM Module) */
static void MX_USART1_UART_Init(void) {
    huart1.Instance = USART1;
    huart1.Init.BaudRate = 115200;
    huart1.Init.WordLength = UART_WORDLENGTH_8B;
    huart1.Init.StopBits = UART_STOPBITS_1;
    huart1.Init.Parity = UART_PARITY_NONE;
    huart1.Init.Mode = UART_MODE_TX_RX;
    huart1.Init.HwFlowCtl = UART_HWCONTROL_NONE;
    huart1.Init.OverSampling = UART_OVERSAMPLING_16;
    HAL_UART_Init(&huart1);
}

/* USART2 Initialization (Debug UART) */
static void MX_USART2_UART_Init(void) {
    huart2.Instance = USART2;
    huart2.Init.BaudRate = 115200;
    huart2.Init.WordLength = UART_WORDLENGTH_8B;
    huart2.Init.StopBits = UART_STOPBITS_1;
    huart2.Init.Parity = UART_PARITY_NONE;
    huart2.Init.Mode = UART_MODE_TX_RX;
    huart2.Init.HwFlowCtl = UART_HWCONTROL_NONE;
    huart2.Init.OverSampling = UART_OVERSAMPLING_16;
    HAL_UART_Init(&huart2);
}

/* GPIO Initialization */
static void MX_GPIO_Init(void) {
    GPIO_InitTypeDef GPIO_InitStruct = {0};

    __HAL_RCC_GPIOD_CLK_ENABLE();
    __HAL_RCC_GPIOA_CLK_ENABLE();
    __HAL_RCC_GPIOB_CLK_ENABLE();

    // Output pins (relays)
    HAL_GPIO_WritePin(GPIOB, GPIO_PIN_3 | GPIO_PIN_4 | GPIO_PIN_5 | GPIO_PIN_6, GPIO_PIN_RESET);

    GPIO_InitStruct.Pin = GPIO_PIN_3 | GPIO_PIN_4 | GPIO_PIN_5 | GPIO_PIN_6;
    GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
    GPIO_InitStruct.Pull = GPIO_NOPULL;
    GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_LOW;
    HAL_GPIO_Init(GPIOB, &GPIO_InitStruct);
}

/* Error Handler */
void Error_Handler(void) {
    __disable_irq();
    while (1) {}
}