import serial
import time

SERIAL_PORT = "COM8"
BAUD_RATE = 115200

def send_at_command(ser, command, timeout=2):
    print(f">>> {command}")  # Print the AT command being sent
    ser.write((command + "\r\n").encode())
    time.sleep(0.1)
    end_time = time.time() + timeout
    response = b""
    while time.time() < end_time:
        if ser.in_waiting:
            response += ser.read(ser.in_waiting)
        if b"OK" in response or b"ERROR" in response or b"+QMT" in response:
            break
        time.sleep(0.1)
    decoded_response = response.decode(errors='ignore')
    print(f"<<< {decoded_response.strip()}")  # Print the response received
    return decoded_response

def read_incoming_messages_forever(ser):
    print("Listening for incoming MQTT messages (press Ctrl+C to exit)...")
    buffer = ""
    try:
        while True:
            if ser.in_waiting:
                data = ser.read(ser.in_waiting).decode(errors='ignore')
                buffer += data

                while "\r\n" in buffer:
                    line, buffer = buffer.split("\r\n", 1)
                    line = line.strip()
                    if line.startswith("+QMTRECV:"):
                        print(f"*** MQTT Message Received: {line}")
                    elif line != "":
                        print(f"*** Other URC: {line}")
            time.sleep(0.1)
    except KeyboardInterrupt:
        print("\nStopped listening due to keyboard interrupt.")

def main():
    with serial.Serial(SERIAL_PORT, BAUD_RATE, timeout=1) as ser:
        print("Setting PDP context...")
        print(send_at_command(ser, 'AT+QICSGP=1,1,"airtelgprs.com","","",1'))

        print("Activating PDP context...")
        print(send_at_command(ser, 'AT+QIACT=1', timeout=10))

        print("Opening MQTT TCP connection...")
        print(send_at_command(ser, 'AT+QMTOPEN=0,"test.mosquitto.org",1883', timeout=10))

        time.sleep(5)

        print("Connecting to MQTT broker...")
        print(send_at_command(ser, 'AT+QMTCONN=0,"client01"', timeout=10))

        time.sleep(5)

        print("Subscribing to topic...")
        print(send_at_command(ser, 'AT+QMTSUB=0,1,"home/sensors/temperature",0', timeout=5))

        print("Publishing message...")
        payload = '{"temp":25.5}'
        payload_len = len(payload)
        pub_cmd = f'AT+QMTPUBEX=0,0,0,0,"home/sensors/temperature",{payload_len},"{payload}"'
        print(send_at_command(ser, pub_cmd, timeout=5))

        # Keep listening for incoming MQTT messages indefinitely
        read_incoming_messages_forever(ser)

        # These lines will only run if you stop the listening loop (Ctrl+C)
        print("Disconnecting MQTT client...")
        print(send_at_command(ser, 'AT+QMTDISC=0', timeout=5))

        print("Deactivating PDP context...")
        print(send_at_command(ser, 'AT+QIDEACT=1', timeout=5))

if __name__ == "__main__":
    main()
